"""
Created on Sep. 17 2019

@author: kanazawa
"""
import numpy as np
import pandas as pd
import math
import matplotlib.pyplot as plt
import chainer
import os
import json
from chainer import Variable
from scipy.stats import norm
from scipy import interpolate
from scipy import integrate
import argparse
import shutil

import net 
import Common.Noise as ns
import Common.FileCreater as ct

MODEL_NAME = "gen_iter.npz"
BASELINE = 0
SINGLE_PEAK = 1
VERTICAL_PEAK = 2
START_DETECTION = 3
END_DETECTION = 4     

def hist2rand(bins, hist):
    assert(len(bins) - 1 == len(hist))
    cdf = np.cumsum(hist)
    cdf = cdf / cdf[-1]
    values = np.random.rand()
    value_bins = np.searchsorted(cdf, values)
    random_from_cdf = bins[value_bins] + (bins[value_bins + 1] - bins[value_bins]) * np.random.rand()
    assert(bins[value_bins] <= random_from_cdf)
    assert(random_from_cdf <= bins[value_bins + 1])
    return random_from_cdf

def ChangePeakWidth(peak, width_point):
    intensity = max(peak) - min(peak) 
    maxIx = np.argmax(peak)
    ts = np.abs(peak[:(maxIx + 1)] - intensity / 2).argmin()
    te = maxIx + np.abs(peak[maxIx:] - intensity / 2).argmin()
    f = interpolate.interp1d(range(0, len(peak)), peak, kind = "cubic")
    peak_range = len(peak) * width_point / (te - ts)
    resamplingIndexes = np.linspace(0, len(peak) - 1, peak_range)
    changePeak = f(resamplingIndexes)
    return changePeak - min(changePeak)

class Chromatogram:
    def __init__(self, peak_GAN_model, param, baseline, time, orgBaseline, orgTime, length = 1024):
        self.length = length
        self.peak_GAN_model = peak_GAN_model
        self.param = param
        self.baseline = np.array(baseline)
        self.org_baseline = np.array(orgBaseline)            
        self.chromatogram_peak_int = np.zeros(self.length)  
        self.chromatogram_int = np.zeros(self.length)       
        self.chromatogram_true_ground = np.zeros(self.length, dtype=int)
        self.chromatogram_time = np.array(time)
        self.chromatogram_org_time = np.array(orgTime)            
        self.r_time_ix = np.array([], dtype = int)
        self.peak_table = pd.DataFrame(columns = ["rTime", "startTime", "endTime", "startInt", "endInt", "True_Area",\
                                                  "error_peak", "error_peak(%)", "error_chromato", "error_chromato(%)", "type", "sn"])

    def GetCenter(self):
        return (min(self.chromatogram_time) + max(self.chromatogram_time)) / 2

    def ChangePeak(self, intensity, width):
        z = np.asarray(self.peak_GAN_model.make_hidden(1))
        peak = np.array(self.peak_GAN_model.forward(z)[0][0].data, dtype = np.float32)
        baseline = [j for j in np.linspace(peak[0],
                                            peak[-1], 
                                            num = len(peak))]
        peak -= baseline
        peak *= intensity
        assert(math.isclose(peak[0], 0.0, abs_tol = 1e-9))
        assert(math.isclose(peak[-1], 0.0, abs_tol = 1e-9))

        peak = peak[peak > 1e-9]
        peak = np.insert(peak, 0, 0.0)
        peak = np.append(peak, 0.0)
        assert(math.isclose(peak[0], 0.0, abs_tol = 1e-9))
        assert(math.isclose(peak[-1], 0.0, abs_tol = 1e-9))

        interval = (max(self.chromatogram_time) - min(self.chromatogram_time)) / self.length
        peak = ChangePeakWidth(peak, round(width / interval))

        return peak  

    def AddFirstPeak(self, pos, intensity, width, sn):
        peak = self.ChangePeak(intensity, width)
        pos_index = np.abs(pos - self.chromatogram_time).argmin()
        startPos = int(pos_index - peak.argmax())
        endPos = startPos + len(peak)

        if not(endPos < 0 or startPos >= self.length):
            assert min(peak) >= 0

            for i in range(startPos, endPos): 
                if(0 <= i < self.length):
                    self.chromatogram_int[i] += peak[i - startPos]
            if(0 <= startPos and endPos <= self.length):
                for i in range(startPos, endPos): 
                    if(0 <= i < self.length):
                        self.chromatogram_peak_int[i] += peak[i - startPos]
                area = integrate.simps(peak, self.chromatogram_time[startPos:endPos])
                for i in range(startPos + 1, endPos - 1):
                    self.chromatogram_true_ground[i] += 1
                r_time = startPos + peak.argmax()
                self.r_time_ix = np.append(self.r_time_ix, r_time)
                s = pd.Series([r_time, 0, 0, 0, 0, area, 0, 0, 0, 0, "", sn], index = self.peak_table.columns, name = "1")
                self.peak_table = self.peak_table.append(s)
            assert 0 == self.chromatogram_true_ground[0]
            assert 0 == self.chromatogram_true_ground[-1]
            return "Done"
        else:
            return "Out Of Range"

    def AddSecondPeak(self, pos, intensity, width, direction, peak_number, sn):
        peak = self.ChangePeak(intensity, width)
        pos_index = np.abs(pos - self.chromatogram_time).argmin()
        startPos = int(pos_index - peak.argmax())
        endPos = startPos + len(peak)

        overlap_index = np.logical_or(SINGLE_PEAK == self.chromatogram_true_ground[startPos:endPos], 
                                      VERTICAL_PEAK == self.chromatogram_true_ground[startPos:endPos])

        if(np.any(True == overlap_index)):
            peaktop_ix = startPos + peak.argmax()
            if("right" == direction):
                left_top = np.max(self.r_time_ix[self.r_time_ix < endPos])
                chromatogram = np.copy(self.chromatogram_int) 
                assert left_top < peaktop_ix
                for i in range(startPos, endPos):
                    if(0 <= i < self.length):
                        chromatogram[i] += peak[i - startPos]
                valley_ix = left_top + np.argmin(chromatogram[left_top:peaktop_ix + 1])
                if(left_top == valley_ix or peaktop_ix == valley_ix):
                    return "Impossible to insert"
        
            if("left" == direction):
                right_top = np.min(self.r_time_ix[startPos <= self.r_time_ix])
                chromatogram = np.copy(self.chromatogram_int) 
                assert peaktop_ix < right_top
                for i in range(startPos, endPos):
                    if(0 <= i < self.length):
                        chromatogram[i] += peak[i - startPos]
                valley_ix = peaktop_ix + np.argmin(chromatogram[peaktop_ix:right_top + 1])
                if(right_top == valley_ix or peaktop_ix == valley_ix):
                    return "Impossible to insert"

        if not(endPos < 0 or startPos >= self.length):
            for i in range(startPos, endPos): 
                if(0 <= i < self.length):
                    self.chromatogram_int[i] += peak[i - startPos]
            if(0 <= startPos and endPos <= self.length):
                for i in range(startPos, endPos): 
                    if(0 <= i < self.length):
                        self.chromatogram_peak_int[i] += peak[i - startPos]
                area = integrate.simps(peak, self.chromatogram_time[startPos:endPos])

                if(np.all(False == overlap_index)):
                    self.chromatogram_true_ground[startPos + 1:endPos - 1] = np.full(endPos - startPos - 2, SINGLE_PEAK)
                    r_time = startPos + peak.argmax()
                    self.r_time_ix = np.append(self.r_time_ix, r_time)
                    s = pd.Series([r_time, 0, 0, 0, 0, area, 0, 0, 0, 0, "", sn], index = self.peak_table.columns, name = peak_number)
                    self.peak_table = self.peak_table.append(s)
                else:
                    left = np.logical_or(SINGLE_PEAK == self.chromatogram_true_ground[startPos:pos_index],
                                         VERTICAL_PEAK == self.chromatogram_true_ground[startPos:pos_index])
                    if(np.any(left)):
                        peak_ix = np.logical_or(SINGLE_PEAK == self.chromatogram_true_ground[:endPos],
                                                VERTICAL_PEAK == self.chromatogram_true_ground[:endPos])  #start <= ix "<=" end
                        peak_ix = np.where(True == peak_ix)[0]
                        border_ix = np.array(0)
                        border_ix = np.append(border_ix, np.where(1 != np.diff(peak_ix))[0] + 1)
                        border_ix = np.append(border_ix, len(peak_ix))
                        single_peak_ranges = []
                        for i in range(len(border_ix) -1):
                            s = border_ix[i]
                            e = border_ix[i + 1]
                            single_peak_ranges.append(peak_ix[s:e])

                        left_start = min(startPos + 1, single_peak_ranges[-1][0])

                        while(SINGLE_PEAK == self.chromatogram_true_ground[endPos - 1]
                                or VERTICAL_PEAK == self.chromatogram_true_ground[endPos - 1]):
                            endPos += 1
                        assert left_start < endPos, "ピークの位置関係が違う" + str(left_start) + str(endPos) 
                        self.chromatogram_true_ground[left_start:endPos - 1] = np.full(endPos - left_start - 1, VERTICAL_PEAK)
                        assert 0 == self.chromatogram_true_ground[0], "不正なラベル"
                        assert 0 == self.chromatogram_true_ground[-1], "不正なラベル" 

                    right = np.logical_or(SINGLE_PEAK == self.chromatogram_true_ground[pos_index:endPos],
                                         VERTICAL_PEAK == self.chromatogram_true_ground[pos_index:endPos])
                    if(np.any(right)):
                        peak_ix = np.logical_or(SINGLE_PEAK == self.chromatogram_true_ground[startPos:], 
                                             VERTICAL_PEAK == self.chromatogram_true_ground[startPos:]) 
                        peak_ix = np.where(True == peak_ix)[0]
                        border_ix = np.array(0)
                        border_ix = np.append(border_ix, np.where(1 != np.diff(peak_ix))[0] + 1)
                        border_ix = np.append(border_ix, len(peak_ix))
                        single_peak_ranges = []
                        for i in range(len(border_ix) -1):
                            s = border_ix[i]
                            e = border_ix[i + 1]
                            single_peak_ranges.append(peak_ix[s:e])
                        right_end = max(endPos - 1, startPos + single_peak_ranges[0][-1] + 1)
                        while(SINGLE_PEAK == self.chromatogram_true_ground[startPos]
                                or VERTICAL_PEAK == self.chromatogram_true_ground[startPos]):
                            startPos -= 1
                        assert startPos < right_end
                        self.chromatogram_true_ground[startPos + 1:right_end] = np.full(right_end - startPos - 1, VERTICAL_PEAK)
                        assert 0 == self.chromatogram_true_ground[0]
                        assert 0 == self.chromatogram_true_ground[-1]

                    r_time = pos_index
                    self.r_time_ix = np.append(self.r_time_ix, r_time)
                    s = pd.Series([r_time, 0, 0, 0, 0, area, 0, 0, 0, 0, "", sn], index = self.peak_table.columns, name = peak_number)
                    self.peak_table = self.peak_table.append(s)                 
            return "Done"
        else:
            return "Impossible to insert"

    def Normalization(self):
        self.chromatogram_int -= min(self.chromatogram_int)
        if(0 != max(self.chromatogram_int)):
            self.chromatogram_int = self.chromatogram_int / max(self.chromatogram_int) * 2 - 1.0 

    def ShowPeak(self):
        peak = np.copy(self.chromatogram_int)      
        plt.plot(self.chromatogram_time, peak)
        plt.xlabel("Time(min)")
        plt.ylabel("Intenisty")
        r_time_list = list(self.r_time_ix)
        plt.plot(self.chromatogram_time[r_time_list], peak[r_time_list], 'o')
        
    def ShowChrom(self):
        print(self.peak_table)  
        plt.plot(self.chromatogram_time, self.chromatogram_int)
        diffIntensities = np.diff(np.r_[self.chromatogram_int, self.chromatogram_int[-1]], n= 1) 
        plt.plot(self.chromatogram_time, diffIntensities)
        plt.xlabel("Time(min)")
        plt.ylabel("Intenisty")
        r_time_list = list(self.r_time_ix)
        plt.plot(self.chromatogram_time[r_time_list], self.chromatogram_int[r_time_list], 'o')
        for index, row in self.peak_table.iterrows():
            s_ix = int(row["startTime"])
            e_ix = int(row["endTime"])
            s_t = self.chromatogram_time[s_ix]
            e_t = self.chromatogram_time[e_ix]
            s_int = row["startInt"]
            e_int = row["endInt"]
            plt.plot([s_t, e_t], [s_int, e_int], \
                label = "Error Peak {0:.3f}%, Error Chromato {1:.3f}%, Error Chromato {2:.3f}, S/N  {3:.3f}".format(row["error_peak(%)"], 
                row["error_chromato(%)"], row["error_chromato"], row["sn"]))
            plt.plot(s_t, s_int, "o")
            plt.plot(e_t, e_int, "o")
        plt.legend(loc='upper right')
        plt.show()

    def ShowChromAndLabel(self):
        plt.plot(self.chromatogram_time, self.chromatogram_int)
        plt.plot(self.chromatogram_time, self.chromatogram_true_ground * 0.1 - 1.5)
        plt.xlabel("Time(min)")
        plt.ylabel("Intenisty")
        r_time_list = list(self.r_time_ix)
        plt.plot(self.chromatogram_time[r_time_list], self.chromatogram_int[r_time_list], 'o')
        plt.show()

    def GeneratePeaks(self):
        num_of_peak = int(round(hist2rand(self.param["num_bins"], self.param["num_hist"])))

        if(0 == num_of_peak):
            return self.chromatogram_int            
        else:
            noise = ns.ASTM(self.chromatogram_time, self.baseline)
            if(math.isclose(noise, 0.0, abs_tol = 1e-9)):
                return
            sn = math.exp(hist2rand(self.param["sn_bins"], self.param["sn_hist"]))
            intensity = sn * noise
            width = hist2rand(self.param["fwhm_bins"], self.param["fwhm_hist"])
            pos = hist2rand(self.param["pos_bins"], self.param["pos_hist"]) + self.GetCenter()
            result = self.AddFirstPeak(pos, intensity, width, sn)
            if("Out Of Range" == result):
                return
            
            num_of_inserted_peak = 1
            if(num_of_inserted_peak >= num_of_peak):
                return

            right_pos = hist2rand(self.param["distance_bins"], self.param["distance_hist"]) + pos
            left_pos = pos - hist2rand(self.param["distance_bins"], self.param["distance_hist"])
            init = np.random.randint(0, 2)     
            while True:
                if(0 == init):
                    sn = math.exp(hist2rand(self.param["sn_bins"], self.param["sn_hist"]))
                    intensity = sn * noise
                    width = hist2rand(self.param["fwhm_bins"], self.param["fwhm_hist"])
                    result = self.AddSecondPeak(right_pos, intensity, width, "right", str(num_of_inserted_peak + 1), sn)
                    if("Done" == result):
                        num_of_inserted_peak += 1
                    right_pos = hist2rand(self.param["distance_bins"], self.param["distance_hist"]) + right_pos
                    if(num_of_inserted_peak >= num_of_peak):        
                        return

                init = 0
                sn = math.exp(hist2rand(self.param["sn_bins"], self.param["sn_hist"]))
                intensity = sn * noise
                width = hist2rand(self.param["fwhm_bins"], self.param["fwhm_hist"])
                result = self.AddSecondPeak(left_pos, intensity, width, "left", str(num_of_inserted_peak + 1), sn)
                if("Done" == result):
                    num_of_inserted_peak += 1
                left_pos = left_pos - hist2rand(self.param["distance_bins"], self.param["distance_hist"])
                if(num_of_inserted_peak >= num_of_peak):         
                    return

                if(left_pos < min(self.chromatogram_time) and max(self.chromatogram_time) < right_pos):         
                    return
    
    def AddPeakTable(self, peak_number, start_ix, end_ix, type, start_int, end_int):
        self.peak_table.at[peak_number, "startTime"] = start_ix
        self.peak_table.at[peak_number, "endTime"] = end_ix
        peak_area = integrate.simps(self.chromatogram_peak_int[start_ix:end_ix + 1], self.chromatogram_time[start_ix:end_ix + 1])
        true_area = self.peak_table.at[peak_number, "True_Area"]
        self.peak_table.at[peak_number, "error_peak"] = peak_area - true_area
        self.peak_table.at[peak_number, "error_peak(%)"] = np.abs(peak_area - true_area) / true_area * 100
        self.peak_table.at[peak_number, "type"] = type
        self.peak_table.at[peak_number, "startInt"] = start_int
        self.peak_table.at[peak_number, "endInt"] = end_int
        baseline = [i for i in np.linspace(start_int, end_int, num = end_ix - start_ix + 1)]
        modifiedPeak = self.chromatogram_peak_int[start_ix:end_ix + 1] + self.baseline[start_ix:end_ix + 1] - baseline
        chro_area = integrate.simps(modifiedPeak, self.chromatogram_time[start_ix:end_ix + 1])
        self.peak_table.at[peak_number, "error_chromato"] = chro_area - true_area
        self.peak_table.at[peak_number, "error_chromato(%)"] = np.abs(chro_area - true_area) / true_area * 100          

    def CreatePeakTabelAndAnnotation(self):
        single_peak_ix = np.where(SINGLE_PEAK == self.chromatogram_true_ground)[0] 
        if(len(single_peak_ix) >= 1):
            border_ix = np.array(0)
            border_ix = np.append(border_ix, np.where(1 != np.diff(single_peak_ix))[0] + 1)
            border_ix = np.append(border_ix, len(single_peak_ix))
            single_peak_ranges = []
            for i in range(len(border_ix) -1):
                s = border_ix[i]
                e = border_ix[i + 1]
                single_peak_ranges.append(single_peak_ix[s:e])
            for single_peak_range in single_peak_ranges:
                if(len(single_peak_range) < 3):
                    return "Invalid"
                self.chromatogram_true_ground[single_peak_range[0] - 1] = START_DETECTION
                self.chromatogram_true_ground[single_peak_range[1] - 1] = START_DETECTION
                self.chromatogram_true_ground[single_peak_range[-2] + 1] = END_DETECTION
                self.chromatogram_true_ground[single_peak_range[-1] + 1] = END_DETECTION
                peak_number = self.peak_table.query("@single_peak_range[0] <= rTime < @single_peak_range[-1]").index
                if(1 != len(peak_number)):
                    return "Invalid"
                peak_number = peak_number[0]
                start_ix = single_peak_range[0]
                end_ix = single_peak_range[-1]
                self.AddPeakTable(peak_number[0], start_ix - 1, end_ix + 1, "Single", self.baseline[start_ix - 1], self.baseline[end_ix + 1])

        vertical_peak_ix = np.where(VERTICAL_PEAK == self.chromatogram_true_ground)[0]
        if(len(vertical_peak_ix) >= 1):
            self.peak_table = self.peak_table.sort_values("rTime")
            border_ix = np.array(0)
            border_ix = np.append(border_ix, np.where(1 != np.diff(vertical_peak_ix))[0] + 1)
            border_ix = np.append(border_ix, len(vertical_peak_ix))
            vertical_peak_ranges = []
            for i in range(len(border_ix) -1):
                s = border_ix[i]
                e = border_ix[i + 1]
                vertical_peak_ranges.append(vertical_peak_ix[s:e])
            for vertical_peak_range in vertical_peak_ranges:
                vertical_peak_range = np.insert(vertical_peak_range, [0, len(vertical_peak_range)], [vertical_peak_range[0] - 1, vertical_peak_range[-1] + 1])
                true_areas = self.peak_table.query("@vertical_peak_range[0] <= rTime < @vertical_peak_range[-1]").True_Area
                start_ix = vertical_peak_range[0]
                end_ix = vertical_peak_range[-1]
                vertical_baseline = [i for i in np.linspace(self.baseline[start_ix], self.baseline[end_ix], num = end_ix - start_ix + 1)]
                cut_start_ix = 0
                cut_end_ix = len(vertical_baseline) - 1 
                for true_area in true_areas[:-1]:
                    areas = np.array([])
                    peak_number = self.peak_table.query("@start_ix <= rTime <= @end_ix").index
                    if(len(peak_number) >= 2):  
                        cut_end_ix  = self.peak_table.at[peak_number[1], "rTime"] - 1 - vertical_peak_range[0]                
                    for ix, i in enumerate(vertical_peak_range[cut_start_ix:cut_end_ix + 1]):
                        if(i - start_ix <= 2):
                            areas = np.append(areas, 0.0)
                            continue
                        v = i + 1
                        modifiedPeak = self.chromatogram_peak_int[start_ix:v]
                        area = integrate.simps(modifiedPeak, self.chromatogram_time[start_ix:v])
                        if(area <= 0):
                            return "NoV"
                        areas = np.append(areas, area)
                    cut_v_index = cut_start_ix + np.abs(true_area - areas).argmin()
                    v_index = vertical_peak_range[cut_v_index]
                    peak_number = self.peak_table.query("@start_ix <= rTime <= @v_index").index
                    if(1 != len(peak_number)):
                        return "Invalid"
                    self.chromatogram_true_ground[start_ix] = START_DETECTION
                    self.chromatogram_true_ground[start_ix + 1] = START_DETECTION
                    self.chromatogram_true_ground[v_index - 1] = END_DETECTION           
                    self.chromatogram_true_ground[v_index] = END_DETECTION
                    peak_number = peak_number[0]
                    self.AddPeakTable(peak_number[0], start_ix, v_index, "Vertical", vertical_baseline[cut_start_ix], vertical_baseline[cut_v_index])
                    cut_start_ix = cut_v_index + 1
                    start_ix = vertical_peak_range[cut_v_index + 1]
                modifiedPeak = self.chromatogram_peak_int[start_ix:end_ix + 1]
                area = integrate.simps(modifiedPeak, self.chromatogram_time[start_ix:end_ix + 1])
                self.chromatogram_true_ground[start_ix] = START_DETECTION
                self.chromatogram_true_ground[start_ix + 1] = START_DETECTION
                self.chromatogram_true_ground[end_ix - 1] = END_DETECTION
                self.chromatogram_true_ground[end_ix] = END_DETECTION      
                peak_number = self.peak_table.query("@start_ix <= rTime <= @end_ix").index
                assert 1 == len(peak_number)
                peak_number = peak_number[0]
                self.AddPeakTable(peak_number[0], start_ix, end_ix, "Vertical End", vertical_baseline[cut_start_ix], vertical_baseline[-1]) 

    def Generate(self):
        self.GeneratePeaks()
        assert min(self.chromatogram_int >= 0)
        result = self.CreatePeakTabelAndAnnotation()
        if("Invalid" == result):
            return "Invalid"
        if("NoV" == result):
            return "NoV"
        self.chromatogram_int += self.baseline
        assert not(np.any(np.isnan(self.chromatogram_int)))

def get_args():
    parser = argparse.ArgumentParser(description='Fake Chromato Generator')
    parser.add_argument('--start', '-st', type=int, default=0)
    parser.add_argument('--end', '-e', type=str, default="10")
    parser.add_argument('--type', '-t', type=str, default="Test")
    parser.add_argument('--seed', '-se', type=int, default = 0)
    parser.add_argument('--number_of_data', '-n', type=int, default = 10)
    return parser.parse_args() 

def main():
    args = get_args()
    np.random.seed(args.seed)
    peak_GAN_model = net.Generator(t_len=701, sigma=4.5) 
    path = os.path.dirname(os.path.abspath(__file__)) + "\\"
    chainer.serializers.load_npz(path + MODEL_NAME, peak_GAN_model)

    param = {}
    param["sn_bins"] = np.load(path + u"Histogram\\SNの一覧_bins.npy")
    param["sn_hist"] = np.load(path + u"Histogram\\SNの一覧_hist.npy")
    param["fwhm_bins"] = np.load(path + u"Histogram\\半値幅の一覧_bins.npy")
    param["fwhm_hist"] = np.load(path + u"Histogram\\半値幅の一覧_hist.npy")
    param["num_bins"] = np.load(path + u"Histogram\\ピーク個数の一覧_bins.npy")
    param["num_hist"] = np.load(path + u"Histogram\\ピーク個数の一覧_hist.npy")
    param["pos_bins"] = np.load(path + u"Histogram\\クロマトの中心付近にあるピークの一覧_bins.npy")
    param["pos_hist"] = np.load(path + u"Histogram\\クロマトの中心付近にあるピークの一覧_hist.npy")
    param["distance_bins"] = np.load(path + u"Histogram\\ピーク間距離の一覧_bins.npy")
    param["distance_hist"] = np.load(path + u"Histogram\\ピーク間距離の一覧_hist.npy")

    number_of = args.number_of_data
    kinds = args.type            
    output_path_unet_base = path + "output\\" + "U-net\\" + kinds + "\\" + str(number_of)  
    if not os.path.isdir(output_path_unet_base):
        os.makedirs(output_path_unet_base)
    shutil.rmtree(output_path_unet_base)

    if("all" == args.end):
        end_ix = number_of
    else:
        end_ix = int(args.end)
    i = args.start
    path_blank = os.path.dirname(os.path.abspath(__file__)) + "\\Data\\" + kinds + "\\Blank_json\\blank.json"
    with open(path_blank, "r") as f:
        baselineDict = json.load(f)
        while end_ix > i:
            dataNum = i + 1
            key = str(i % len(baselineDict) + 1)
            global num_of_peak
            num_of_peak = dataNum
            print(num_of_peak)
            orgBaseline = baselineDict[key]["orgIntensity"]
            orgTime = baselineDict[key]["orgTime"]
            chrom = Chromatogram(peak_GAN_model, param, orgBaseline, 
                                orgTime, orgBaseline, orgTime, len(orgTime))
            result = chrom.Generate()
            if("NoV" == result or "Invalid" == result):
                i -= 1 
                continue
 
            print('Please enter output file name.')
            filename = input()
            output_path_unet = output_path_unet_base + "\\" + filename
            if not os.path.isdir(output_path_unet):
                os.makedirs(output_path_unet)
            funet_input = open(output_path_unet + "\\" + filename + r"_input.txt", 'w')
            funet_output = open(output_path_unet + "\\" + filename + r"_output.txt", 'w')      

            ct.CreateChromatoTextFile(funet_input, 
                                    chrom.chromatogram_time,
                                    chrom.chromatogram_int, 
                                    filename,
                                    "compound# ")
            ct.CreateChromatoTextFile(funet_output, 
                                    chrom.chromatogram_time, 
                                    chrom.chromatogram_true_ground, 
                                    filename,
                                    "compound# ")
    
            funet_input.close()
            funet_output.close()
            i += 1

if __name__ == '__main__':
    main()